/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: $
   Last Modification:
    $Author: Robert $
    $Modtime: $
    $Revision: 12210 $

   Targets:
     rcX          : yes

   Description:
    rcX OS Abstraction Implementation

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         15.01.2011  SS    intitial version

**************************************************************************************/

/* rcX includes */
#include <rX_Includes.h>

/* Standard includes */
#include <stdio.h>

#include "..\ToolkitSample.h"
#include "cifXErrors.h"

/*****************************************************************************/
/*! O/S Specific initialization (initializes libpciaccess)
*     \return CIFX_NO_ERROR on success                                       */
/*****************************************************************************/
long OS_Init(void)
{
  return CIFX_NO_ERROR;
}

/*****************************************************************************/
/*! O/S Specific de-initialization (de-initializes libpciaccess)             */
/*****************************************************************************/
void OS_Deinit(void)
{
}

/*****************************************************************************/
/*! Memory allocation wrapper (standard malloc)
*     \param ulSize Size of block to allocate
*     \return NULL on failure                                                */
/*****************************************************************************/
void* OS_Memalloc(uint32_t ulSize)
{
  void* pvMem = NULL;
  (void)rX_MemAllocateMemory(&pvMem, ulSize);
  return pvMem;
}

/*****************************************************************************/
/*! Memory de-allocation wrapper (standard free)
*     \param pvMem  Block to free                                            */
/*****************************************************************************/
void OS_Memfree(void* pvMem)
{
  (void)rX_MemFreeMemory(pvMem);
}

/*****************************************************************************/
/*! Memory resize wrapper (standard realloc)
*     \param pvMem      Block to resize
*     \param ulNewSize  New size of the block
*     \return NULL on error                                                  */
/*****************************************************************************/
void* OS_Memrealloc(void* pvMem, uint32_t ulNewSize)
{
  void* pvNewMem = NULL;
  (void)rX_MemReAllocateMemory(&pvMem, &pvNewMem, ulNewSize);
  return pvNewMem;
}

/*****************************************************************************/
/*! Memset wrapper
*     \param pvMem   Memory to set
*     \param bFill   Fill byte
*     \param ulSize  Size of the fill block                                  */
/*****************************************************************************/
void OS_Memset(void* pvMem, uint8_t bFill, uint32_t ulSize)
{
  memset(pvMem, bFill, ulSize);
}

/*****************************************************************************/
/*! Memcopy wrapper
*     \param pvDest  Destination pointer
*     \param pvSrc   Source pointer
*     \param ulSize  Size to copy                                            */
/*****************************************************************************/
void OS_Memcpy(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memcpy(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Memcompare wrapper
*     \param pvBuf1  First compare buffer
*     \param pvBuf2  Second compare buffer
*     \param ulSize  Size to compare
*     \return 0 if blocks are equal                                          */
/*****************************************************************************/
int OS_Memcmp(void* pvBuf1, void* pvBuf2, uint32_t ulSize)
{
  return memcmp(pvBuf1, pvBuf2, ulSize);
}

/*****************************************************************************/
/*! Memmove wrapper (Overlapping memory copy)
*     \param pvDest  Destination buffer
*     \param pvSrc   Source buffer
*     \param ulSize  Size to move                                            */
/*****************************************************************************/
void OS_Memmove(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memmove(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Read PCI configuration area of specified card
*     \param pvOSDependent OS Dependent parameter to identify card
*     \return Pointer to configuration data (passed to WritePCIConfig)       */
/*****************************************************************************/
void* OS_ReadPCIConfig(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  return NULL;  /*lint !e438 */
}

/*****************************************************************************/
/*! Restore PCI configuration
*     \param pvOSDependent OS Dependent parameter to identify card
*     \param pvPCIConfig   Pointer returned from ReadPCIConfig               */
/*****************************************************************************/
void OS_WritePCIConfig(void* pvOSDependent, void* pvPCIConfig)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  UNREFERENCED_PARAMETER(pvPCIConfig);
} /*lint !e438 */

static void cifXTKitISR(RX_HANDLE hInt, void* pvPrm)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvPrm;
  if (CIFX_TKIT_IRQ_DSR_REQUESTED == cifXTKitISRHandler(&ptrcXInst->tDevice, FALSE))
  {
    cifXTKitDSRHandler(&ptrcXInst->tDevice);
  }
#if defined(__NETX51)
  *((volatile unsigned long*)Adr_NX51_hif_pio_irq_raw) |= MSK_NX51_hif_pio_irq_raw_irq_pio47;
#endif
  return;
}

/*****************************************************************************/
/*! Enable interrupts on the given device
*     \param pvOSDependent Pointer to internal device structure              */
/*****************************************************************************/
void OS_EnableInterrupts(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvOSDependent;
  
  if(RX_OK == Drv_IntIdentifyInterrupt("HIF", 0, &ptrcXInst->hIrq))
  { 
#if defined(__NETX51)
    *((volatile unsigned long*)Adr_NX51_hif_pio_irq_arm_mask_set) |= MSK_NX51_hif_pio_irq_arm_mask_set_irq_pio47;
#endif
    (void)Drv_IntInitializeInterrupt(ptrcXInst->hIrq, cifXTKitISR, ptrcXInst);
    (void)Drv_IntEnableInterrupt(ptrcXInst->hIrq);
  } else
  {
    USER_Trace(pvOSDependent, TRACE_LEVEL_ERROR, "Attach interrupt failed. Using polling mode");
    ptrcXInst->tDevice.fIrqEnabled = FALSE;
  }    
}

/*****************************************************************************/
/*! Disable interrupts on the given device
*     \param pvOSDependent Pointer to internal device structure              */
/*****************************************************************************/
void OS_DisableInterrupts(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvOSDependent;
  (void)Drv_IntDisableInterrupt(ptrcXInst->hIrq);
}

/*****************************************************************************/
/*! Open file for reading
*     \param szFilename   File to open (including path)
*     \param pulFileSize  Returned size of the file in bytes
*     \return Handle to the file, NULL on failure                            */
/*****************************************************************************/
void* OS_FileOpen(char* szFile, uint32_t* pulFileLen)
{
  UNREFERENCED_PARAMETER(szFile);
  UNREFERENCED_PARAMETER(pulFileLen);
  return NULL; /*lint !e438 */
}

/*****************************************************************************/
/*! Read data from file
*     \param pvFile    Handle to the file (acquired by OS_FileOpen)
*     \param ulOffset  Offset to read from
*     \param ulSize    Size to read
*     \param pvBuffer  Buffer to read data into
*     \return number of bytes read                                           */
/*****************************************************************************/
uint32_t OS_FileRead(void* pvFile, uint32_t ulOffset, uint32_t ulSize, void* pvBuffer)
{
  UNREFERENCED_PARAMETER(pvFile);
  UNREFERENCED_PARAMETER(ulOffset);  
  UNREFERENCED_PARAMETER(ulSize);
  UNREFERENCED_PARAMETER(pvBuffer);
  return 0; /*lint !e438 */
}

/*****************************************************************************/
/*! Close open file
*     \param pvFile    Handle to the file (acquired by OS_FileOpen)          */
/*****************************************************************************/
void OS_FileClose(void* pvFile)
{
  UNREFERENCED_PARAMETER(pvFile);
} /*lint !e438 */

/*****************************************************************************/
/*! Get Millisecond counter value (used for timeout handling)
*     \return Counter value with a resolution of 1ms                         */
/*****************************************************************************/
uint32_t OS_GetMilliSecCounter(void)
{
  return (rX_SysGetSystemTicks()*(rX_SysGetSystemCycletime()/1000));
}

/*****************************************************************************/
/*! Sleep for the given time
*     \param ulSleepTimeMs Time in ms to sleep (0 will sleep for 50us)       */
/*****************************************************************************/
void OS_Sleep(uint32_t ulSleepTimeMs)
{
  if (0 == ulSleepTimeMs)
    return;
  else 
    (void)rX_SysSleepTask(ulSleepTimeMs);
}

/*****************************************************************************/
/*! Create Lock (Usually same as mutex, but does not support timed waiting)
*     \return Handle to created lock                                         */
/*****************************************************************************/
void* OS_CreateLock(void)
{
  return OS_CreateMutex();
}

/*****************************************************************************/
/*! Acquire a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_EnterLock(void* pvLock)
{
  (void)OS_WaitMutex(pvLock, RX_INFINITE);
}

/*****************************************************************************/
/*! Release a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_LeaveLock(void* pvLock)
{
  (void)OS_ReleaseMutex(pvLock);
}

/*****************************************************************************/
/*! Delete a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_DeleteLock(void* pvLock)
{
  (void)OS_DeleteMutex(pvLock);
}

/*****************************************************************************/
/*! Create mutex
*     \return Handle to new created mutex                                    */
/*****************************************************************************/
void* OS_CreateMutex(void)
{
  void* pvMutex = OS_Memalloc(RX_MUTEX_SIZE);
  
  (void)OS_Memset(pvMutex, 0x00, RX_MUTEX_SIZE);
  if (RX_OK != rX_MtxCreateMutex(NULL, pvMutex, 2))
  {
    (void)OS_Memfree(pvMutex);
    pvMutex = NULL;
  }
  
  return pvMutex; 
}

/*****************************************************************************/
/*! Try to acquire mutex with timeout
*     \param pvMutex   Handle to mutex
*     \param ulTimeout Timeout in ms to wait for mutex
*     \return !=0 if mutex was acquired                                      */
/*****************************************************************************/
int OS_WaitMutex(void* pvMutex, uint32_t ulTimeout)
{
  int iRet = 0;
  
  if (RX_OK == rX_MtxLockMutex(pvMutex, ulTimeout))
  {
    iRet = 1;
  }
  
  return iRet;  
}

/*****************************************************************************/
/*! Release previously acquired mutex
*     \param pvMutex   Handle to mutex                                       */
/*****************************************************************************/
void OS_ReleaseMutex(void* pvMutex)
{
  (void)rX_MtxUnlockMutex(pvMutex);
}

/*****************************************************************************/
/*! Delete mutex
*     \param pvMutex   Handle to mutex                                       */
/*****************************************************************************/
void OS_DeleteMutex(void* pvMutex)
{
  (void)rX_MtxDeleteMutex(pvMutex);
  (void)OS_Memfree(pvMutex);
}

/*****************************************************************************/
/*! Create event
*     \return Handle to created event                                        */
/*****************************************************************************/
void* OS_CreateEvent(void)
{
  void* pvEvent = OS_Memalloc(RX_SEMAPHORE_SIZE);

  (void)OS_Memset(pvEvent, 0x00, RX_SEMAPHORE_SIZE);
  if (RX_OK != rX_SemCreateSemaphore(NULL, pvEvent, 2))
  {
    OS_Memfree(pvEvent);
    pvEvent = NULL;
  }

  return pvEvent; 
}

/*****************************************************************************/
/*! Signal event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_SetEvent(void* pvEvent)
{
  (void)rX_SemPutSemaphore(pvEvent);
}

/*****************************************************************************/
/*! Reset event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_ResetEvent(void* pvEvent)
{
  (void)rX_SemClearSemaphoreCount(pvEvent);
}

/*****************************************************************************/
/*! Delete event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_DeleteEvent(void* pvEvent)
{
  (void)rX_SemDeleteSemaphore(pvEvent);
  (void)OS_Memfree(pvEvent);
}

/*****************************************************************************/
/*! Wait for event
*     \param pvEvent   Handle to event
*     \param ulTimeout Timeout in ms to wait for event
*     \return CIFX_EVENT_SIGNALLED if event was set, CIFX_EVENT_TIMEOUT otherwise */
/*****************************************************************************/
uint32_t OS_WaitEvent(void* pvEvent, uint32_t ulTimeout)
{
  uint32_t ulRet = CIFX_EVENT_TIMEOUT;
  
  if (RX_OK == rX_SemWaitForSemaphore(pvEvent, ulTimeout))
  {
    ulRet = CIFX_EVENT_SIGNALLED;
  }
  
  return ulRet; 
}

/*****************************************************************************/
/*! Map driver pointer to user space
*     \param pvDriverMem  Pointer to driver memory
*     \param ulMemSize    Size of the memory to map
*     \param ppvMappedMem Returned mapped pointer
*     \param os_dependent OS Dependent parameter in DEVICEINSTANCE
*     \return Handle to mapping, NULL on error                               */
/*****************************************************************************/
void* OS_MapUserPointer(void* pvDriverMem, uint32_t ulMemSize, void** ppvMappedMem, void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(ulMemSize);
  UNREFERENCED_PARAMETER(pvOSDependent);
  /* We are running in user mode, so it is not necessary to map anything to user space */ 
  *ppvMappedMem = pvDriverMem;
  
  return pvDriverMem; /*lint !e438 */
}

/*****************************************************************************/
/*! Unmap previously mapped user space pointer 
*     \param phMapping  Handle returned from OS_MapUserPointer
*     \param os_dependent OS Dependent parameter in DEVICEINSTANCE
*     \return 0 on error                                                     */
/*****************************************************************************/
int OS_UnmapUserPointer(void* phMapping, void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(phMapping);
  UNREFERENCED_PARAMETER(pvOSDependent);
  /* We are running in user mode, so it is not necessary to map anything to user space */ 
  return 1; /*lint !e438 */
}

/*****************************************************************************/
/*! Compare strings
*     \param pszBuf1  String buffer 1
*     \param pszBuf2  String buffer 2
*     \return 0 if strings are equal                                         */
/*****************************************************************************/
int OS_Strcmp(const char* pszBuf1, const char* pszBuf2)
{
  return strcmp(pszBuf1, pszBuf2);
}

/*****************************************************************************/
/*! Compare strings case insensitive
*     \param pszBuf1  String buffer 1
*     \param pszBuf2  String buffer 2
*     \param ulLen    Maximum length to compare
*     \return 0 if strings are equal                                         */
/*****************************************************************************/
int OS_Strnicmp(const char* pszBuf1, const char* pszBuf2, uint32_t ulLen)
{
  return strnicmp(pszBuf1, pszBuf2, ulLen);
}

/*****************************************************************************/
/*! Get length of string
*     \param szText  Text buffer
*     \return Length of given string                                         */
/*****************************************************************************/
int OS_Strlen(const char* szText)
{
  return strlen(szText);
}

/*****************************************************************************/
/*! Copy string to destination buffer
*     \param szText   Destination string
*     \param szSource Source string
*     \param ulLen    Maximum length to copy
*     \return Pointer to szDest                                              */
/*****************************************************************************/
char* OS_Strncpy(char* szDest, const char* szSource, uint32_t ulLen)
{
  return strncpy(szDest, szSource, ulLen);
}

#ifdef CIFX_TOOLKIT_ENABLE_DSR_LOCK
static volatile int s_iIrq;
/*****************************************************************************/
/*! This functions needs to provide a lock against the interrupt service 
*   routine of the device. The easiest way is an IRQ lock but some operating 
*   systems provide a way to lock against a specific interrupt
*     \param pvODDependent OS Dependent parameter in DEVICEINSTANCE          */
/*****************************************************************************/
void OS_IrqLock(void* pvOSDependent)
{
  NX50_lock_irqfiq_save(s_iIrq);   
}

/*****************************************************************************/
/*! This function re-enables the device's interrupt service routine.
*     \param pvODDependent OS Dependent parameter in DEVICEINSTANCE          */
/*****************************************************************************/
void OS_IrqUnlock(void* pvOSDependent)
{
  NX50_lock_irqfiq_restore(s_iIrq);
}
#endif
